'use client';

import React, { useState, useEffect } from 'react';
import { 
  Plus, 
  Calendar, 
  Search, 
  ChevronRight, 
  ChevronLeft, 
  Save, 
  Trash2, 
  FileText, 
  Paperclip, 
  Users, 
  CheckCircle2,
  Clock,
  LayoutDashboard,
  Briefcase,
  ShieldCheck,
  X
} from 'lucide-react';
import { motion, AnimatePresence } from 'motion/react';

// --- Types ---
type Department = 'آی تی' | 'اداری' | 'بازرگانی' | 'مالی' | 'مدیریت' | 'سایر';
type Priority = 'کم' | 'متوسط' | 'زیاد' | 'فوری';

interface ProjectFile {
  id: string;
  name: string;
  url: string;
}

interface ProjectStage {
  id: string;
  title: string;
  description: string;
  createdAt: string;
  updatedAt: string;
  nextActionDate: string;
  managerNote: string;
  assignedDept: Department;
  assignedUser: string;
}

interface Project {
  id: string;
  code: string; // TIA-001
  department: Department;
  creatorName: string;
  creatorCode: string;
  priority: Priority;
  subject: string;
  description: string;
  documents: string;
  files: ProjectFile[];
  createdAt: string;
  stages: ProjectStage[];
}

interface DailyReportRow {
  id: string;
  rowNum: number;
  hour: string;
  plannedWork: string;
  projectIds: string[]; // IDs of selected projects
  tasks: string;
  createdAt: string;
  updatedAt: string;
  managerNote: string;
}

interface DailyReport {
  id: string;
  date: string; // YYYY-MM-DD
  generalNote: string;
  rows: DailyReportRow[];
}

// --- Mock Data ---
const DEPARTMENTS: Department[] = ['آی تی', 'اداری', 'بازرگانی', 'مالی', 'مدیریت', 'سایر'];
const USERS = [
  { id: '1', name: 'بصیری', dept: 'مالی', code: '1001' },
  { id: '2', name: 'احمدی', dept: 'آی تی', code: '1002' },
  { id: '3', name: 'رضایی', dept: 'بازرگانی', code: '1003' },
];

// --- Utilities ---
const getJalaliDate = (date: Date = new Date()) => {
  return new Intl.DateTimeFormat('fa-IR-u-ca-persian', {
    dateStyle: 'full',
  }).format(date);
};

const getJalaliDateTime = (date: Date = new Date()) => {
  return new Intl.DateTimeFormat('fa-IR-u-ca-persian', {
    dateStyle: 'short',
    timeStyle: 'short',
  }).format(date);
};

const formatTIA = (num: number) => `TIA-${String(num).padStart(3, '0')}`;

export default function TIAApp() {
  const [activeTab, setActiveTab] = useState<'reports' | 'projects' | 'manager'>('reports');
  const [currentUser] = useState({ name: 'بصیری', code: '1001', role: 'manager' });
  
  // State
  const [projects, setProjects] = useState<Project[]>([]);
  const [dailyReports, setDailyReports] = useState<DailyReport[]>([]);
  const [selectedDate, setSelectedDate] = useState(new Date().toISOString().split('T')[0]);
  const [searchQuery, setSearchQuery] = useState('');
  
  // Modals
  const [isProjectSelectorOpen, setIsProjectSelectorOpen] = useState(false);
  const [activeReportRowId, setActiveReportRowId] = useState<string | null>(null);

  // --- Handlers ---
  
  // Project Handlers
  const handleAddProject = () => {
    const nextCodeNum = projects.length + 1;
    const newProject: Project = {
      id: Math.random().toString(36).substr(2, 9),
      code: formatTIA(nextCodeNum),
      department: 'آی تی',
      creatorName: currentUser.name,
      creatorCode: currentUser.code,
      priority: 'متوسط',
      subject: '',
      description: '',
      documents: '',
      files: [],
      createdAt: getJalaliDateTime(),
      stages: []
    };
    setProjects([newProject, ...projects]);
  };

  const updateProject = (id: string, updates: Partial<Project>) => {
    setProjects(projects.map(p => p.id === id ? { ...p, ...updates } : p));
  };

  const addStage = (projectId: string) => {
    const stage: ProjectStage = {
      id: Math.random().toString(36).substr(2, 9),
      title: '',
      description: '',
      createdAt: getJalaliDateTime(),
      updatedAt: getJalaliDateTime(),
      nextActionDate: '',
      managerNote: '',
      assignedDept: 'آی تی',
      assignedUser: ''
    };
    setProjects(projects.map(p => 
      p.id === projectId ? { ...p, stages: [...p.stages, stage] } : p
    ));
  };

  const updateStage = (projectId: string, stageId: string, updates: Partial<ProjectStage>) => {
    setProjects(projects.map(p => {
      if (p.id !== projectId) return p;
      return {
        ...p,
        stages: p.stages.map(s => s.id === stageId ? { ...s, ...updates, updatedAt: getJalaliDateTime() } : s)
      };
    }));
  };

  // Daily Report Handlers
  const currentReport = dailyReports.find(r => r.date === selectedDate) || {
    id: Math.random().toString(36).substr(2, 9),
    date: selectedDate,
    generalNote: '',
    rows: []
  };

  const handleAddRow = () => {
    if (currentReport.rows.length >= 9) return;
    
    const startHour = 7 + currentReport.rows.length;
    const newRow: DailyReportRow = {
      id: Math.random().toString(36).substr(2, 9),
      rowNum: currentReport.rows.length + 1,
      hour: `${startHour} تا ${startHour + 1}`,
      plannedWork: '',
      projectIds: [],
      tasks: '',
      createdAt: getJalaliDateTime(),
      updatedAt: getJalaliDateTime(),
      managerNote: ''
    };

    const updatedReport = {
      ...currentReport,
      rows: [...currentReport.rows, newRow]
    };

    setDailyReports(prev => {
      const filtered = prev.filter(r => r.date !== selectedDate);
      return [...filtered, updatedReport];
    });
  };

  const updateRow = (rowId: string, updates: Partial<DailyReportRow>) => {
    const updatedRows = currentReport.rows.map(r => 
      r.id === rowId ? { ...r, ...updates, updatedAt: getJalaliDateTime() } : r
    );
    const updatedReport = { ...currentReport, rows: updatedRows };
    setDailyReports(prev => {
      const filtered = prev.filter(r => r.date !== selectedDate);
      return [...filtered, updatedReport];
    });
  };

  const toggleProjectSelection = (projectId: string) => {
    if (!activeReportRowId) return;
    const row = currentReport.rows.find(r => r.id === activeReportRowId);
    if (!row) return;

    const newProjectIds = row.projectIds.includes(projectId)
      ? row.projectIds.filter(id => id !== projectId)
      : [...row.projectIds, projectId];
    
    updateRow(activeReportRowId, { projectIds: newProjectIds });
  };

  return (
    <div className="min-h-screen bg-black text-white p-4 md:p-8 font-sans" dir="rtl">
      {/* Header */}
      <header className="max-w-7xl mx-auto mb-8 flex flex-col md:flex-row justify-between items-center gap-4 border-b border-gold pb-6">
        <div>
          <h1 className="text-3xl font-bold text-gold mb-2">سیستم جامع مدیریت TIA</h1>
          <p className="text-gray-400 flex items-center gap-2">
            <Users size={18} className="text-gold" />
            کاربر: {currentUser.name} ({currentUser.code}) | {getJalaliDate()}
          </p>
        </div>
        
        <nav className="flex gap-2">
          <button 
            onClick={() => setActiveTab('reports')}
            className={`dsr-button flex items-center gap-2 ${activeTab === 'reports' ? 'bg-white' : ''}`}
          >
            <LayoutDashboard size={20} /> گزارش روزانه
          </button>
          <button 
            onClick={() => setActiveTab('projects')}
            className={`dsr-button flex items-center gap-2 ${activeTab === 'projects' ? 'bg-white' : ''}`}
          >
            <Briefcase size={20} /> ثبت پروژه
          </button>
          <button 
            onClick={() => setActiveTab('manager')}
            className={`dsr-button flex items-center gap-2 ${activeTab === 'manager' ? 'bg-white' : ''}`}
          >
            <ShieldCheck size={20} /> پنل مدیریت
          </button>
        </nav>
      </header>

      <main className="max-w-7xl mx-auto">
        <AnimatePresence mode="wait">
          {/* --- Daily Reports Tab --- */}
          {activeTab === 'reports' && (
            <motion.div 
              key="reports"
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              exit={{ opacity: 0, y: -20 }}
              className="space-y-6"
            >
              <div className="flex flex-wrap items-center justify-center gap-4 mb-8">
                <button 
                  onClick={() => {
                    const d = new Date(selectedDate);
                    d.setDate(d.getDate() - 1);
                    setSelectedDate(d.toISOString().split('T')[0]);
                  }}
                  className="p-2 border border-gold rounded-full hover:bg-gold hover:text-black transition-colors"
                >
                  <ChevronRight size={24} />
                </button>
                <div className="dsr-card py-2 px-8 flex items-center gap-4">
                  <Calendar className="text-gold" />
                  <input 
                    type="date" 
                    value={selectedDate}
                    onChange={(e) => setSelectedDate(e.target.value)}
                    className="bg-transparent text-gold font-bold text-xl outline-none cursor-pointer"
                  />
                </div>
                <button 
                  onClick={() => {
                    const d = new Date(selectedDate);
                    d.setDate(d.getDate() + 1);
                    setSelectedDate(d.toISOString().split('T')[0]);
                  }}
                  className="p-2 border border-gold rounded-full hover:bg-gold hover:text-black transition-colors"
                >
                  <ChevronLeft size={24} />
                </button>
              </div>

              <div className="overflow-x-auto">
                <table className="dsr-table">
                  <thead>
                    <tr>
                      <th>ردیف</th>
                      <th>ساعت</th>
                      <th>کارها</th>
                      <th>پروژه‌ها</th>
                      <th>وظایف</th>
                      <th>بروزرسانی</th>
                      <th>نظر مدیر</th>
                    </tr>
                  </thead>
                  <tbody>
                    {currentReport.rows.map((row) => (
                      <tr key={row.id}>
                        <td>{row.rowNum}</td>
                        <td>{row.hour}</td>
                        <td>
                          <textarea 
                            value={row.plannedWork}
                            onChange={(e) => updateRow(row.id, { plannedWork: e.target.value })}
                            className="dsr-input min-h-[60px] resize-none"
                          />
                        </td>
                        <td>
                          <button 
                            onClick={() => {
                              setActiveReportRowId(row.id);
                              setIsProjectSelectorOpen(true);
                            }}
                            className="text-xs border border-gold p-1 rounded hover:bg-gold hover:text-black transition-all w-full"
                          >
                            {row.projectIds.length > 0 
                              ? `${row.projectIds.length} پروژه انتخاب شده`
                              : "انتخاب پروژه"}
                          </button>
                          <div className="mt-1 flex flex-wrap gap-1 justify-center">
                            {row.projectIds.map(pid => {
                              const p = projects.find(proj => proj.id === pid);
                              return p ? (
                                <span key={pid} className="text-[10px] bg-gold/20 text-gold px-1 rounded border border-gold/30">
                                  {p.code}
                                </span>
                              ) : null;
                            })}
                          </div>
                        </td>
                        <td>
                          <textarea 
                            value={row.tasks}
                            onChange={(e) => updateRow(row.id, { tasks: e.target.value })}
                            className="dsr-input min-h-[60px] resize-none"
                          />
                        </td>
                        <td className="text-xs text-gray-400">{row.updatedAt}</td>
                        <td className="text-xs italic text-gold">{row.managerNote || "—"}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>

              <div className="flex justify-between items-start gap-8 mt-8">
                <div className="flex-1 dsr-card">
                  <h4 className="text-gold mb-2 flex items-center gap-2"><FileText size={18}/> توضیحات کلی روز</h4>
                  <textarea 
                    value={currentReport.generalNote}
                    onChange={(e) => {
                      const updatedReport = { ...currentReport, generalNote: e.target.value };
                      setDailyReports(prev => {
                        const filtered = prev.filter(r => r.date !== selectedDate);
                        return [...filtered, updatedReport];
                      });
                    }}
                    placeholder="توضیحات اضافی برای کل روز..."
                    className="dsr-input h-32"
                  />
                </div>
                <button 
                  onClick={handleAddRow}
                  disabled={currentReport.rows.length >= 9}
                  className="dsr-button flex items-center gap-2 h-fit mt-8 disabled:opacity-50"
                >
                  <Plus size={20} /> افزودن سطر جدید
                </button>
              </div>
            </motion.div>
          )}

          {/* --- Project Registration Tab --- */}
          {activeTab === 'projects' && (
            <motion.div 
              key="projects"
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              exit={{ opacity: 0, y: -20 }}
              className="space-y-8"
            >
              <div className="flex justify-between items-center">
                <h2 className="text-2xl font-bold text-gold">مدیریت پروژه‌ها</h2>
                <button onClick={handleAddProject} className="dsr-button flex items-center gap-2">
                  <Plus size={20} /> ثبت پروژه جدید
                </button>
              </div>

              <div className="grid gap-6">
                {projects.map((project) => (
                  <div key={project.id} className="dsr-card border-2">
                    <div className="flex flex-wrap justify-between items-center mb-6 border-b border-gold/30 pb-4">
                      <div className="flex items-center gap-4">
                        <span className="text-3xl font-black text-gold">{project.code}</span>
                        <div className="text-xs text-gray-400">
                          ایجاد شده در: {project.createdAt} <br />
                          توسط: {project.creatorName} ({project.creatorCode})
                        </div>
                      </div>
                      <div className="flex gap-4">
                        <div className="flex flex-col">
                          <label className="text-[10px] text-gold uppercase">بخش سازمانی</label>
                          <select 
                            value={project.department}
                            onChange={(e) => updateProject(project.id, { department: e.target.value as Department })}
                            className="dsr-input py-1"
                          >
                            {DEPARTMENTS.map(d => <option key={d} value={d}>{d}</option>)}
                          </select>
                        </div>
                        <div className="flex flex-col">
                          <label className="text-[10px] text-gold uppercase">اولویت</label>
                          <select 
                            value={project.priority}
                            onChange={(e) => updateProject(project.id, { priority: e.target.value as Priority })}
                            className="dsr-input py-1"
                            disabled={currentUser.role !== 'manager'}
                          >
                            <option value="کم">کم</option>
                            <option value="متوسط">متوسط</option>
                            <option value="زیاد">زیاد</option>
                            <option value="فوری">فوری</option>
                          </select>
                        </div>
                      </div>
                    </div>

                    <div className="grid md:grid-cols-2 gap-6 mb-6">
                      <div className="space-y-4">
                        <div>
                          <label className="block text-sm text-gold mb-1">موضوع پروژه</label>
                          <input 
                            type="text" 
                            value={project.subject}
                            onChange={(e) => updateProject(project.id, { subject: e.target.value })}
                            className="dsr-input"
                            placeholder="عنوان کوتاه پروژه..."
                          />
                        </div>
                        <div>
                          <label className="block text-sm text-gold mb-1">شرح و اهداف</label>
                          <textarea 
                            value={project.description}
                            onChange={(e) => updateProject(project.id, { description: e.target.value })}
                            className="dsr-input h-32"
                            placeholder="جزئیات کامل پروژه را اینجا بنویسید..."
                          />
                        </div>
                      </div>
                      <div className="space-y-4">
                        <div>
                          <label className="block text-sm text-gold mb-1">مستندات و مدارک</label>
                          <textarea 
                            value={project.documents}
                            onChange={(e) => updateProject(project.id, { documents: e.target.value })}
                            className="dsr-input h-20 mb-2"
                            placeholder="توضیحات مدارک..."
                          />
                          <div className="border border-dashed border-gold/50 rounded-lg p-4 text-center">
                            <button className="text-gold flex items-center gap-2 mx-auto hover:underline">
                              <Plus size={16} /> افزودن فایل جدید
                            </button>
                            <p className="text-[10px] text-gray-500 mt-1">فایل‌های آپلود شده در اینجا نمایش داده می‌شوند</p>
                          </div>
                        </div>
                      </div>
                    </div>

                    {/* Stages Section */}
                    <div className="mt-8 border-t border-gold/20 pt-6">
                      <div className="flex justify-between items-center mb-4">
                        <h3 className="text-lg font-bold text-gold flex items-center gap-2">
                          <Clock size={20} /> مراحل و فازهای پروژه
                        </h3>
                        <button 
                          onClick={() => addStage(project.id)}
                          className="text-xs bg-gold text-black px-3 py-1 rounded font-bold hover:bg-white transition-all"
                        >
                          + افزودن مرحله
                        </button>
                      </div>

                      <div className="space-y-4">
                        {project.stages.map((stage, index) => (
                          <div key={stage.id} className="bg-white/5 border border-gold/20 rounded-lg p-4 relative">
                            <div className="absolute top-4 left-4 text-[10px] text-gray-500 text-left">
                              ثبت: {stage.createdAt} <br />
                              بروزرسانی: {stage.updatedAt}
                            </div>
                            <div className="flex items-center gap-2 mb-4">
                              <span className="bg-gold text-black w-6 h-6 rounded-full flex items-center justify-center font-bold text-xs">
                                {index + 1}
                              </span>
                              <input 
                                type="text"
                                value={stage.title}
                                onChange={(e) => updateStage(project.id, stage.id, { title: e.target.value })}
                                placeholder="عنوان مرحله..."
                                className="bg-transparent border-b border-gold/30 outline-none text-gold font-bold flex-1"
                              />
                            </div>
                            
                            <div className="grid md:grid-cols-3 gap-4">
                              <div className="md:col-span-2">
                                <label className="block text-[10px] text-gold mb-1">شرح اقدامات صورت گرفته</label>
                                <textarea 
                                  value={stage.description}
                                  onChange={(e) => updateStage(project.id, stage.id, { description: e.target.value })}
                                  className="dsr-input h-24 text-sm"
                                />
                              </div>
                              <div className="space-y-3">
                                <div>
                                  <label className="block text-[10px] text-gold mb-1">تاریخ اقدام بعدی</label>
                                  <input 
                                    type="text"
                                    value={stage.nextActionDate}
                                    onChange={(e) => updateStage(project.id, stage.id, { nextActionDate: e.target.value })}
                                    placeholder="۱۴۰۲/۱۲/۰۱"
                                    className="dsr-input text-sm"
                                  />
                                </div>
                                <div>
                                  <label className="block text-[10px] text-gold mb-1">ارجاع به (بخش / فرد)</label>
                                  <div className="flex gap-1">
                                    <select 
                                      value={stage.assignedDept}
                                      onChange={(e) => updateStage(project.id, stage.id, { assignedDept: e.target.value as Department })}
                                      className="dsr-input py-1 text-[10px]"
                                    >
                                      {DEPARTMENTS.map(d => <option key={d} value={d}>{d}</option>)}
                                    </select>
                                    <select 
                                      value={stage.assignedUser}
                                      onChange={(e) => updateStage(project.id, stage.id, { assignedUser: e.target.value })}
                                      className="dsr-input py-1 text-[10px]"
                                    >
                                      <option value="">انتخاب فرد</option>
                                      {USERS.filter(u => u.dept === stage.assignedDept).map(u => (
                                        <option key={u.id} value={u.name}>{u.name}</option>
                                      ))}
                                    </select>
                                  </div>
                                </div>
                              </div>
                            </div>

                            <div className="mt-4 pt-4 border-t border-gold/10">
                              <label className="block text-[10px] text-gold mb-1">نظر مدیر</label>
                              <textarea 
                                value={stage.managerNote}
                                onChange={(e) => updateStage(project.id, stage.id, { managerNote: e.target.value })}
                                className="dsr-input h-12 text-xs italic border-dashed"
                                disabled={currentUser.role !== 'manager'}
                                placeholder="فقط مدیر می‌تواند در اینجا بنویسد..."
                              />
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </motion.div>
          )}

          {/* --- Manager Panel Tab --- */}
          {activeTab === 'manager' && (
            <motion.div 
              key="manager"
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              exit={{ opacity: 0, y: -20 }}
              className="space-y-6"
            >
              <div className="dsr-card flex flex-wrap gap-4 items-end">
                <div className="flex-1 min-w-[200px]">
                  <label className="block text-sm text-gold mb-1">فیلتر کاربر</label>
                  <select className="dsr-input">
                    <option value="">همه کاربران</option>
                    {USERS.map(u => <option key={u.id} value={u.id}>{u.name} ({u.dept})</option>)}
                  </select>
                </div>
                <div className="flex-1 min-w-[200px]">
                  <label className="block text-sm text-gold mb-1">فیلتر تاریخ</label>
                  <input type="date" className="dsr-input" />
                </div>
                <button className="dsr-button">اعمال فیلتر</button>
                <button className="dsr-button bg-red-600 text-white hover:bg-red-700">حذف گروهی</button>
              </div>

              <div className="overflow-x-auto">
                <table className="dsr-table">
                  <thead>
                    <tr>
                      <th>کاربر</th>
                      <th>ردیف</th>
                      <th>ساعت</th>
                      <th>کارها</th>
                      <th>پروژه‌ها</th>
                      <th>وظایف</th>
                      <th>تاریخ ثبت</th>
                      <th>نظر مدیر</th>
                      <th>عملیات</th>
                    </tr>
                  </thead>
                  <tbody>
                    {/* Manager rows would be populated here */}
                    <tr>
                      <td colSpan={9} className="py-12 text-gray-500">دیتایی برای نمایش وجود ندارد. لطفاً فیلتر کنید.</td>
                    </tr>
                  </tbody>
                </table>
              </div>
            </motion.div>
          )}
        </AnimatePresence>
      </main>

      {/* --- Project Selector Modal --- */}
      <AnimatePresence>
        {isProjectSelectorOpen && (
          <div className="fixed inset-0 z-50 flex items-center justify-center p-4">
            <motion.div 
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              exit={{ opacity: 0 }}
              onClick={() => setIsProjectSelectorOpen(false)}
              className="absolute inset-0 bg-black/80 backdrop-blur-sm"
            />
            <motion.div 
              initial={{ scale: 0.9, opacity: 0 }}
              animate={{ scale: 1, opacity: 1 }}
              exit={{ scale: 0.9, opacity: 0 }}
              className="dsr-card w-full max-w-2xl relative z-10 max-h-[80vh] overflow-hidden flex flex-col"
            >
              <div className="flex justify-between items-center mb-4 border-b border-gold pb-2">
                <h3 className="text-xl font-bold text-gold">انتخاب پروژه</h3>
                <button onClick={() => setIsProjectSelectorOpen(false)} className="text-gold hover:text-white">
                  <X size={24} />
                </button>
              </div>
              
              <div className="mb-4 relative">
                <Search className="absolute right-3 top-1/2 -translate-y-1/2 text-gold" size={18} />
                <input 
                  type="text" 
                  placeholder="جستجو در کد یا عنوان پروژه..."
                  className="dsr-input pr-10"
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                />
              </div>

              <div className="overflow-y-auto flex-1 space-y-2 pr-2">
                {projects.filter(p => 
                  p.code.includes(searchQuery) || p.subject.includes(searchQuery)
                ).map(p => {
                  const isSelected = activeReportRowId && 
                    currentReport.rows.find(r => r.id === activeReportRowId)?.projectIds.includes(p.id);
                  
                  return (
                    <div 
                      key={p.id}
                      onClick={() => toggleProjectSelection(p.id)}
                      className={`p-3 rounded-lg border cursor-pointer transition-all flex justify-between items-center ${
                        isSelected ? 'border-gold bg-gold/20' : 'border-white/10 hover:border-gold/50'
                      }`}
                    >
                      <div>
                        <span className="font-black text-gold mr-2">{p.code}</span>
                        <span className="text-sm">{p.subject || "(بدون عنوان)"}</span>
                      </div>
                      {isSelected && <CheckCircle2 size={18} className="text-gold" />}
                    </div>
                  );
                })}
                {projects.length === 0 && (
                  <p className="text-center py-8 text-gray-500">هیچ پروژه‌ای ثبت نشده است.</p>
                )}
              </div>

              <div className="mt-6 pt-4 border-t border-gold/30 flex justify-end">
                <button 
                  onClick={() => setIsProjectSelectorOpen(false)}
                  className="dsr-button"
                >
                  تایید و بستن
                </button>
              </div>
            </motion.div>
          </div>
        )}
      </AnimatePresence>

      {/* Toast Simulation */}
      <div className="fixed bottom-8 right-8 pointer-events-none">
        {/* Toast notifications could go here */}
      </div>
    </div>
  );
}
